<?php
require_once 'db.php';
check_login();

if ($_SESSION['role'] !== 'admin') {
    exit('アクセス権限がありません。');
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['backup_type'])) {
    $type = $_POST['backup_type']; // 'full', 'sql_only', 'img_only'

    try {
        $zipname = 'backup_' . $type . '_' . date('Ymd_His') . '.zip';
        $tmp_zip = tempnam(sys_get_temp_dir(), 'zip');
        $zip = new ZipArchive();

        if ($zip->open($tmp_zip, ZipArchive::CREATE) !== TRUE) {
            exit("ZIPファイルの作成に失敗しました。");
        }

        // --- SQL作成（full または sql_only） ---
        if ($type === 'full' || $type === 'sql_only') {
            $tables = [];
            $result = $pdo->query("SHOW TABLES");
            while ($row = $result->fetch(PDO::FETCH_NUM)) { $tables[] = $row[0]; }

            $sql_content = "-- バックアップ: " . date("Y-m-d H:i:s") . "\nSET FOREIGN_KEY_CHECKS=0;\n";
            foreach ($tables as $table) {
                $sql_content .= "\nDROP TABLE IF EXISTS `$table`;\n";
                $res = $pdo->query("SHOW CREATE TABLE `$table` ");
                $r = $res->fetch(PDO::FETCH_NUM);
                $sql_content .= $r[1] . ";\n\n";

                $res = $pdo->query("SELECT * FROM `$table` ");
                while ($r = $res->fetch(PDO::FETCH_NUM)) {
                    $sql_content .= "INSERT INTO `$table` VALUES(" . implode(',', array_map(function($v) use ($pdo) {
                        return is_null($v) ? 'NULL' : $pdo->quote($v);
                    }, $r)) . ");\n";
                }
            }
            $sql_content .= "\nSET FOREIGN_KEY_CHECKS=1;\n";
            $zip->addFromString('database_backup.sql', $sql_content);
        }

        // --- 画像フォルダ追加（full または img_only） ---
        if ($type === 'full' || $type === 'img_only') {
            $upload_dir = 'uploads';
            if (is_dir($upload_dir)) {
                $files = new RecursiveIteratorIterator(
                    new RecursiveDirectoryIterator($upload_dir, RecursiveDirectoryIterator::SKIP_DOTS),
                    RecursiveIteratorIterator::LEAVES_ONLY
                );
                foreach ($files as $file) {
                    if (!$file->isDir()) {
                        $filePath = $file->getRealPath();
                        $relativePath = 'uploads/' . substr($filePath, strlen(realpath($upload_dir)) + 1);
                        $zip->addFile($filePath, $relativePath);
                    }
                }
            }
        }

        $zip->close();

        header('Content-Type: application/zip');
        header('Content-Disposition: attachment; filename="' . $zipname . '"');
        header('Content-Length: ' . filesize($tmp_zip));
        ob_clean();
        flush();
        readfile($tmp_zip);
        unlink($tmp_zip);
        exit;

    } catch (Exception $e) {
        $error_msg = "エラー: " . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <title>データバックアップ出力</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css">
    <style>
        .backup-header { background: linear-gradient(135deg, #198754 0%, #146c43 100%); }
        .manual-box { background-color: #fff; border: 1px dashed #ced4da; }
        #loading-overlay {
            display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%;
            background: rgba(255, 255, 255, 0.9); z-index: 9999;
            flex-direction: column; align-items: center; justify-content: center;
        }
    </style>
</head>
<body class="bg-light">

<div id="loading-overlay">
    <div class="spinner-border text-success mb-3" style="width: 3rem; height: 3rem;" role="status"></div>
    <h5 class="fw-bold text-success">ファイルを圧縮しています...</h5>
    <p class="text-muted small">完了までそのままお待ちください。</p>
</div>

<div class="container mt-5" style="max-width: 750px;">
    <nav aria-label="breadcrumb" class="mb-3">
        <a href="index.php" class="text-decoration-none small text-secondary">← 管理者パネルへ戻る</a>
    </nav>

    <div class="card shadow border-0 overflow-hidden mb-5">
        <div class="card-header backup-header text-white py-3">
            <h5 class="mb-0 fw-bold"><i class="bi bi-database-fill-down me-2"></i>データバックアップ出力</h5>
        </div>
        <div class="card-body p-4">
            
            <?php if (isset($error_msg)): ?>
                <div class="alert alert-danger mb-4"><?= htmlspecialchars($error_msg) ?></div>
            <?php endif; ?>

            <div class="alert alert-info border-0 bg-light mb-4 shadow-sm">
                <h6 class="fw-bold small text-primary"><i class="bi bi-info-circle-fill me-1"></i> バックアップの推奨手順</h6>
                <p class="small mb-0 text-muted">
                    通常は<strong>「フルバックアップ」</strong>で一括保存してください。画像が数千枚ある等、エラー（真っ白や失敗）になる場合は、SQLと画像を別々に保存してください。
                </p>
            </div>

            <form method="POST" id="backupForm">
                <div class="d-grid gap-3">
                    <button type="submit" name="backup_type" value="full" class="btn btn-success btn-lg fw-bold shadow py-3 backup-btn">
                        <i class="bi bi-archive-fill me-2"></i>フルバックアップ（DB + 画像）
                    </button>
                    
                    <div class="row g-2">
                        <div class="col-md-6">
                            <button type="submit" name="backup_type" value="sql_only" class="btn btn-outline-success fw-bold backup-btn w-100">
                                <i class="bi bi-filetype-sql me-2"></i>DBのみ（SQL）
                            </button>
                        </div>
                        <div class="col-md-6">
                            <button type="submit" name="backup_type" value="img_only" class="btn btn-outline-success fw-bold backup-btn w-100">
                                <i class="bi bi-images me-2"></i>画像フォルダのみ
                            </button>
                        </div>
                    </div>
                </div>
            </form>

            <div class="manual-box p-3 mt-4 rounded shadow-sm">
                <h6 class="fw-bold small"><i class="bi bi-cloud-download me-1"></i> バックアップが失敗する場合</h6>
                <p class="small text-muted mb-0">
                    サーバーの容量制限等により、この画面からの画像保存ができないことがあります。その場合は、<strong>FTPソフト等を用いてサーバー上の <code>uploads/</code> フォルダ内を直接手動でダウンロード</strong>してください。
                </p>
            </div>
        </div>
    </div>
</div>

<script>
document.getElementById('backupForm').addEventListener('submit', function(e) {
    const btn = document.activeElement;
    if (!btn || !btn.classList.contains('backup-btn')) return;
    
    if (!confirm('バックアップ作成を開始しますか？\n※完了まで画面を閉じないでください。')) {
        e.preventDefault();
        return;
    }
    document.getElementById('loading-overlay').style.display = 'flex';
    
    // ダウンロード開始後は画面遷移しないため、一旦10秒で隠す
    setTimeout(() => {
        document.getElementById('loading-overlay').style.display = 'none';
    }, 10000);
});
</script>
</body>
</html>