<?php
/*
Plugin Name: My Access Counter (Simple Retro)
Description: 昔懐かしいデザインのアクセスカウンターです。[access_count]で設置出来ます。
Version: 1.0
Author: HATABOW
*/

if (!defined('ABSPATH')) exit;

class MyAccessCounter {

    private $option_count  = 'my_total_views';
    private $option_digits = 'my_counter_digits';
    private $option_comma  = 'my_counter_comma';

    public function __construct() {
        // カウント・管理画面・ショートコードの登録
        add_action('template_redirect', array($this, 'count_visit'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_shortcode('access_count', array($this, 'display_shortcode'));
        add_action('wp_head', array($this, 'add_css'));
    }

    // 共通デザインの適用
    public function add_css() {
        ?>
        <style>
            .retro-counter {
                display: inline-block;
                background-color: #000;
                color: #00ff00;
                font-family: 'Courier New', Courier, monospace;
                font-weight: bold;
                padding: 3px 10px;
                border: 2px inset #666;
                letter-spacing: 2px;
                text-shadow: 0 0 5px #00ff00;
                line-height: 1.2;
                vertical-align: middle;
                border-radius: 2px;
            }
        </style>
        <?php
    }

    // カウント処理（ロボット・管理者・二重カウントを除外）
    public function count_visit() {
        if ($this->is_robot()) return;
        if (current_user_can('edit_posts')) return;
        if (isset($_COOKIE['my_visit_token'])) return;

        $count = get_option($this->option_count, 0);
        update_option($this->option_count, $count + 1);

        // 24時間のクッキーをセット
        setcookie('my_visit_token', '1', time() + 86400, COOKIEPATH, COOKIE_DOMAIN);
    }

    private function is_robot() {
        if (empty($_SERVER['HTTP_USER_AGENT'])) return true;
        $ua = $_SERVER['HTTP_USER_AGENT'];
        $robots = array('Googlebot', 'Yahoo!', 'msnbot', 'bingbot', 'Baiduspider', 'YandexBot', 'Twitterbot', 'facebookexternalhit');
        foreach ($robots as $robot) {
            if (stripos($ua, $robot) !== false) return true;
        }
        return false;
    }

    // ショートコード [access_count] の表示
    public function display_shortcode() {
        $count  = get_option($this->option_count, 0);
        $digits = get_option($this->option_digits, 0);
        $comma  = get_option($this->option_comma, 'yes');

        // 1. 桁数処理
        $display_count = ($digits > 0) ? str_pad($count, $digits, '0', STR_PAD_LEFT) : $count;

        // 2. カンマ処理
        if ($comma === 'yes') {
            if ($digits > 0) {
                $display_count = $this->format_with_commas($display_count);
            } else {
                $display_count = number_format($count);
            }
        }

        return '<span class="retro-counter">' . esc_html($display_count) . '</span>';
    }

    private function format_with_commas($str) {
        $rev = strrev($str);
        $chunks = str_split($rev, 3);
        $formatted = implode(',', $chunks);
        return strrev($formatted);
    }

    // 管理画面の作成
    public function add_admin_menu() {
        add_options_page('アクセスカウンター設定', 'アクセスカウンター', 'manage_options', 'my-access-counter', array($this, 'admin_page'));
    }

    public function admin_page() {
        if (isset($_POST['update_settings']) && check_admin_referer('update_action')) {
            update_option($this->option_count, intval($_POST['new_count']));
            update_option($this->option_digits, intval($_POST['new_digits']));
            update_option($this->option_comma, $_POST['new_comma']);
            echo '<div class="updated"><p>設定を更新しました。</p></div>';
        }

        $count  = get_option($this->option_count, 0);
        $digits = get_option($this->option_digits, 0);
        $comma  = get_option($this->option_comma, 'yes');
        ?>
        <div class="wrap">
            <h1>アクセスカウンター設定</h1>
            <div class="card" style="max-width: 600px; padding: 20px;">
                <form method="post">
                    <?php wp_nonce_field('update_action'); ?>
                    <table class="form-table">
                        <tr>
                            <th>現在のアクセス数</th>
                            <td><input type="number" name="new_count" value="<?php echo esc_attr($count); ?>" class="regular-text"></td>
                        </tr>
                        <tr>
                            <th>表示桁数</th>
                            <td><input type="number" name="new_digits" value="<?php echo esc_attr($digits); ?>" min="0" class="small-text"> (0で自動桁)</td>
                        </tr>
                        <tr>
                            <th>カンマ区切り</th>
                            <td>
                                <label><input type="radio" name="new_comma" value="yes" <?php checked($comma, 'yes'); ?>> あり</label>
                                <label style="margin-left:20px;"><input type="radio" name="new_comma" value="no" <?php checked($comma, 'no'); ?>> なし</label>
                            </td>
                        </tr>
                    </table>
                    <p class="submit"><input type="submit" name="update_settings" class="button button-primary" value="設定を保存"></p>
                </form>
            </div>

            <div style="margin-top: 30px;">
                <h3>プレビュー</h3>
                <div style="padding: 20px; background: #333; display: inline-block; border-radius: 4px;">
                    <?php echo $this->display_shortcode(); ?>
                </div>
                <p>設置コード: <code>[access_count]</code></p>
            </div>
        </div>
        <?php
    }
}
new MyAccessCounter();
